
import cv2
import os
import numpy as np
from PyQt5.QtWidgets import QWidget, QLabel, QVBoxLayout, QHBoxLayout, QTextEdit, QPushButton, QFileDialog, QListWidget, QListWidgetItem, QMessageBox, QDialog, QDialogButtonBox
from PyQt5.QtGui import QImage, QPixmap, QFont
from PyQt5.QtCore import Qt, QTimer, QSettings

class AnaliseWindow(QWidget):
    def __init__(self, caminho1=None, caminho2=None):
        super().__init__()
        self.setWindowTitle("Análise de Vídeo")
        self.setWindowState(Qt.WindowMaximized)

        self.settings = QSettings("VillaReplay", "Config")
        self.theme = self.settings.value("theme", "light")
        self.apply_theme()

        self.mode = 0
        self.zoom_factor = 1.0
        self.zoom_center = None
        self.last_frame1 = None
        self.last_frame2 = None
        self.cap1 = None
        self.cap2 = None

        self.label_cam1 = QLabel()
        self.label_cam2 = QLabel()
        for label in [self.label_cam1, self.label_cam2]:
            label.setAlignment(Qt.AlignCenter)
            label.setStyleSheet("background-color: black;")
            label.setSizePolicy(label.sizePolicy().Expanding, label.sizePolicy().Expanding)

        self.layout_videos = QHBoxLayout()
        self.layout_videos.setContentsMargins(0, 0, 0, 0)
        if caminho1 and caminho2:
            self.layout_videos.addWidget(self.label_cam1, stretch=1)
            self.layout_videos.addWidget(self.label_cam2, stretch=1)
        elif caminho1:
            self.layout_videos.addWidget(self.label_cam1, stretch=1)

        self.status = QLabel()
        self.status.setAlignment(Qt.AlignRight)
        self.status.setFont(QFont("Arial", 10))
        self.status.setStyleSheet("padding: 4px;")

        self.reset_zoom_btn = QPushButton("🔄 Resetar Zoom")
        self.reset_zoom_btn.setFixedHeight(28)
        self.reset_zoom_btn.setCursor(Qt.PointingHandCursor)
        self.reset_zoom_btn.clicked.connect(self.reset_zoom)

        self.abrir_videos_btn = QPushButton("📂 Abrir Arquivos Gravados")
        self.abrir_videos_btn.setFixedHeight(28)
        self.abrir_videos_btn.setCursor(Qt.PointingHandCursor)
        self.abrir_videos_btn.clicked.connect(self.abrir_arquivos)

        self.reabrir_btn = QPushButton("🎞️ Reabrir Análise Gravada")
        self.reabrir_btn.setFixedHeight(28)
        self.reabrir_btn.setCursor(Qt.PointingHandCursor)
        self.reabrir_btn.clicked.connect(self.selecionar_analise_existente)

        self.legenda = QTextEdit()
        self.legenda.setReadOnly(True)
        self.legenda.setFont(QFont("Arial", 11))
        self.legenda.setStyleSheet("background-color: transparent; border: none;")
        self.legenda.setText(
            "Controles:\n"
            "[Espaço] Play/Pause   [←/→] 1 Quadro   [↑/↓] Velocidade\n"
            "[S] Salvar quadro   [R] Resetar Zoom   [🖱️ Scroll] Zoom\n"
            "[1/2/0] Alternar câmera   [Esc] Fechar"
        )

        self.buttons_layout = QHBoxLayout()
        self.buttons_layout.addWidget(self.reset_zoom_btn)
        self.buttons_layout.addWidget(self.abrir_videos_btn)
        self.buttons_layout.addWidget(self.reabrir_btn)

        layout = QVBoxLayout()
        layout.addLayout(self.layout_videos, stretch=5)
        layout.addWidget(self.status)
        layout.addLayout(self.buttons_layout)
        layout.addWidget(self.legenda)
        self.setLayout(layout)

        self.playing = True
        self.setFocusPolicy(Qt.StrongFocus)
        self.setFocus()

        if caminho1:
            self.carregar_videos(caminho1, caminho2)

    def carregar_videos(self, caminho1, caminho2=None):
        self.cap1 = cv2.VideoCapture(caminho1)
        self.cap2 = cv2.VideoCapture(caminho2) if caminho2 else None
        if self.cap2:
            self.total_frames = int(min(self.cap1.get(cv2.CAP_PROP_FRAME_COUNT), self.cap2.get(cv2.CAP_PROP_FRAME_COUNT)))
        else:
            self.total_frames = int(self.cap1.get(cv2.CAP_PROP_FRAME_COUNT))
        self.fps = int(self.cap1.get(cv2.CAP_PROP_FPS)) or 15
        self.current_frame = 0

        self.timer = QTimer()
        self.timer.timeout.connect(self.next_frame)
        self.timer.start(1000 // self.fps)

    def closeEvent(self, event):
        if self.cap1 and hasattr(self.cap1, 'release') and self.cap1.isOpened():
            self.cap1.release()
        if self.cap2 and hasattr(self.cap2, 'release') and self.cap2.isOpened():
            self.cap2.release()
        event.accept()

    def apply_theme(self):
        if self.theme == "dark":
            self.setStyleSheet("background-color: #1c1c1c; color: white;")
        else:
            self.setStyleSheet("background-color: #eaeaea; color: black;")

    def reset_zoom(self):
        self.zoom_center = None
        self.zoom_factor = 1.0
        self.redraw_last_frames()

    def redraw_last_frames(self):
        if self.mode in (0, 1) and self.last_frame1 is not None:
            self.show_frame(self.last_frame1, self.label_cam1)
        if self.mode in (0, 2) and self.cap2 and self.last_frame2 is not None:
            self.show_frame(self.last_frame2, self.label_cam2)

    def abrir_arquivos(self):
        paths, _ = QFileDialog.getOpenFileNames(self, "Selecione os arquivos de vídeo", "", "Vídeos (*.avi *.mp4)")
        if len(paths) == 2:
            if self.cap1:
                self.cap1.release()
            if self.cap2:
                self.cap2.release()
            self.carregar_videos(paths[0], paths[1])
        elif len(paths) == 1:
            if self.cap1:
                self.cap1.release()
            self.carregar_videos(paths[0], None)

    
    def selecionar_analise_existente(self):
        pasta = os.path.abspath("analises")
        if not os.path.exists(pasta):
            QMessageBox.information(self, "Análises", "Nenhuma análise foi gravada ainda.")
            return

        arquivos = sorted(os.listdir(pasta))
        opcoes = {}

        for nome in arquivos:
            if "_Cam1_" in nome:
                base = nome.replace("_Cam1_", "")
                cam2 = nome.replace("_Cam1_", "_Cam2_")
                if cam2 in arquivos:
                    chave = base.replace(".avi", "").replace("_", " ")
                    opcoes[chave] = (
                        os.path.join(pasta, nome),
                        os.path.join(pasta, cam2)
                    )
                else:
                    chave = base.replace(".avi", "").replace("_", " ") + " (apenas Cam1)"
                    opcoes[chave] = (
                        os.path.join(pasta, nome),
                        None
                    )

        if not opcoes:
            QMessageBox.information(self, "Análises", "Nenhuma análise encontrada.")
            return

        lista = QListWidget()
        for chave in sorted(opcoes.keys()):
            item = QListWidgetItem(chave)
            lista.addItem(item)

        def carregar_selecionado():
            sel = lista.currentItem()
            if sel:
                caminho1, caminho2 = opcoes[sel.text()]
                if self.cap1:
                    self.cap1.release()
                if self.cap2:
                    self.cap2.release()
                self.carregar_videos(caminho1, caminho2)
                self.playing = True
                dlg.accept()

        dlg = QDialog(self)
        dlg.setWindowTitle("Selecionar Análise Gravada")
        dlg.setMinimumWidth(400)
        dlg_layout = QVBoxLayout()
        dlg_layout.addWidget(lista)
        btns = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        btns.accepted.connect(carregar_selecionado)
        btns.rejected.connect(dlg.reject)
        dlg_layout.addWidget(btns)
        dlg.setLayout(dlg_layout)
        dlg.exec_()



    def next_frame(self):
        if not self.playing:
            return
        self.step(1)

    def step(self, direction):
        self.current_frame = max(0, min(self.current_frame + direction, self.total_frames - 1))
        self.cap1.set(cv2.CAP_PROP_POS_FRAMES, self.current_frame)
        if self.cap2:
            self.cap2.set(cv2.CAP_PROP_POS_FRAMES, self.current_frame)
        ret1, frame1 = self.cap1.read()
        ret2, frame2 = (self.cap2.read() if self.cap2 else (False, None))
        if ret1:
            self.last_frame1 = frame1
            if self.mode in (0, 1):
                self.show_frame(frame1, self.label_cam1)
        if ret2:
            self.last_frame2 = frame2
            if self.mode in (0, 2):
                self.show_frame(frame2, self.label_cam2)

        elapsed = self.current_frame / self.fps
        total = self.total_frames / self.fps
        self.status.setText(
            f"Frame: {self.current_frame} / {self.total_frames}    Tempo: {self.format_time(elapsed)} / {self.format_time(total)}"
        )

    def format_time(self, seconds):
        m, s = divmod(int(seconds), 60)
        return f"{m:02d}:{s:02d}"


    def show_frame(self, frame, label):
        if frame is None:
            return
        rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)

        if self.zoom_factor > 1.0:
            h, w, _ = rgb.shape
            new_w = int(w / self.zoom_factor)
            new_h = int(h / self.zoom_factor)
            center_x = w // 2
            center_y = h // 2

            if self.zoom_center:
                rel_x = self.zoom_center.x() / self.width()
                rel_y = self.zoom_center.y() / self.height()
                center_x = int(w * rel_x)
                center_y = int(h * rel_y)

            x1 = max(0, min(center_x - new_w // 2, w - new_w))
            y1 = max(0, min(center_y - new_h // 2, h - new_h))
            rgb = rgb[y1:y1 + new_h, x1:x1 + new_w]

        h, w, ch = rgb.shape
        bytes_per_line = ch * w
        qimg = QImage(rgb.tobytes(), w, h, bytes_per_line, QImage.Format_RGB888)
        pixmap = QPixmap.fromImage(qimg)
        label.setPixmap(pixmap.scaled(label.width(), label.height(), Qt.KeepAspectRatio))


    def keyPressEvent(self, event):
        key = event.key()
        if key == Qt.Key_Space:
            self.playing = not self.playing
        elif key == Qt.Key_Left:
            self.step(-1)
        elif key == Qt.Key_Right:
            self.step(1)
        elif key == Qt.Key_Up:
            self.timer.setInterval(max(5, self.timer.interval() - 5))
        elif key == Qt.Key_Down:
            self.timer.setInterval(self.timer.interval() + 5)
        elif key == Qt.Key_S:
            self.export_current_frame()
        elif key == Qt.Key_R:
            self.reset_zoom()
        elif key == Qt.Key_1:
            self.set_mode(1)
        elif key == Qt.Key_2:
            self.set_mode(2)
        elif key == Qt.Key_0:
            self.set_mode(0)
        elif key == Qt.Key_Escape:
            self.close()

    def set_mode(self, mode):
        self.mode = mode
        self.zoom_center = None
        self.zoom_factor = 1.0
        self.label_cam1.setVisible(mode in (0, 1))
        if self.cap2:
            self.label_cam2.setVisible(mode in (0, 2))
        self.redraw_last_frames()

    def export_current_frame(self):
        if self.last_frame1 is not None:
            cv2.imwrite(f"frame_cam1_{self.current_frame}.png", self.last_frame1)
        if self.cap2 and self.last_frame2 is not None:
            cv2.imwrite(f"frame_cam2_{self.current_frame}.png", self.last_frame2)


    def wheelEvent(self, event):
        self.zoom_center = event.pos()
        delta = event.angleDelta().y()
        if delta > 0:
            self.zoom_factor = min(self.zoom_factor + 0.1, 3.0)
        else:
            self.zoom_factor = max(self.zoom_factor - 0.1, 1.0)
        self.redraw_last_frames()
